<?php

namespace App\Livewire\Dashboard\User;

use App\Models\Company;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Livewire\Component;

class Save extends Component
{
    public $companies;
    public $user_id;
    public $name;
    public $email;
    public $password;
    public $password_confirmation;
    public $role = 'cliente';
    public $company_id;
    public $user;
    public $is_super_admin;

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                Rule::unique('users', 'email')->ignore($this->user_id), // Ignorar el correo del usuario actual
            ],
            'password' => 'nullable|string|min:8|confirmed',
            'role' => 'required|in:super_admin,admin,cliente',
            'company_id' => 'nullable|exists:companies,id',
        ];
    }

    public function mount(?int $id = null)
    {
        $this->is_super_admin = Auth::user()->role === 'super_admin';

        if ($id !== null) {
            $this->user = User::findOrFail($id);

            // Verificar acceso a la compañía del usuario
            if (!$this->is_super_admin && Auth::user()->company_id !== $this->user->company_id) {
                session()->flash('error', 'No tienes acceso a este usuario.');
                return redirect()->route('d-user-index');
            }

            $this->user_id = $this->user->id;
            $this->name = $this->user->name;
            $this->email = $this->user->email;
            $this->role = $this->user->role;
            $this->company_id = $this->user->company_id;
        } else {
            $this->company_id = Auth::user()->company_id; // Asignar el company_id del usuario logueado
        }
    }

    public function render()
    {
        $this->companies = Company::all();

        $is_super_admin = $this->is_super_admin;
        return view('livewire.dashboard.user.save', [
            'is_super_admin' => $is_super_admin,
            'companies' => $this->companies,
        ])->layout('layouts.app');
    }

    public function submit()
    {
        // Modificar las reglas dependiendo del contexto
        if (!$this->is_super_admin) {
            $this->rules['role'] = 'in:cliente'; // El rol debe ser cliente si no es super_admin
        }

        $this->validate();

        if ($this->user) {
            $this->user->update([
                'name' => $this->name,
                'email' => $this->email,
                'role' => $this->is_super_admin ? $this->role : 'cliente',
                'password' => $this->password ? Hash::make($this->password) : $this->user->password,
                'company_id' => $this->is_super_admin ? $this->company_id : Auth::user()->company_id,
            ]);
            session()->flash('success', 'Usuario actualizado con éxito.');
        } else {
            User::create([
                'name' => $this->name,
                'email' => $this->email,
                'role' => $this->is_super_admin ? $this->role : 'cliente',
                'password' => Hash::make($this->password),
                'company_id' => $this->is_super_admin ? $this->company_id : Auth::user()->company_id,
            ]);
            session()->flash('success', 'Usuario creado con éxito.');
        }

        return redirect()->route('d-user-index');
    }
}
