<?php
namespace App\Livewire\Dashboard\Payment;

use Illuminate\Support\Facades\Auth;
use App\Models\Payment;
use App\Models\Client;
use Livewire\Component;

class Save extends Component
{
    public $client_id;
    public $amount_paid = 0;
    public $payment_date;
    public $paymet_detail;
    public $payment;
    public $client;

    protected $rules = [
        'client_id' => 'required|exists:clients,id',
        'amount_paid' => 'required|numeric|min:0',
        'payment_date' => 'required|date',
        'paymet_detail' => 'nullable|string|in:EFECTIVO,YAPE',
    ];

    public function mount(?int $id = null)
    {
        if ($id) {
            $this->payment = Payment::with('client')->findOrFail($id);
            $this->client_id = $this->payment->client_id;
            $this->amount_paid = $this->payment->amount_paid;
            $this->payment_date = $this->payment->payment_date;
            $this->paymet_detail = $this->payment->paymet_detail;

            // Validar acceso del usuario si no es super_admin
            if (Auth::user()->role !== 'super_admin' && Auth::user()->company_id !== $this->payment->client->user->company_id) {
                session()->flash('error', 'No tienes acceso a esta página.');
                return redirect()->route('d-client-index');
            }
        }
    }

    public function updatedClientId()
    {
        if ($this->client_id) {
            $client = Client::find($this->client_id);

            // Validar acceso del cliente si no es super_admin
            if ($client && (Auth::user()->role === 'super_admin' || Auth::user()->company_id === $client->user->company_id)) {
                $this->amount_paid = $client->daily_payment;
            } else {
                $this->amount_paid = 0;
                session()->flash('error', 'Cliente no válido o acceso denegado.');
            }
        } else {
            $this->amount_paid = 0; // Resetear si no se selecciona cliente
        }
    }

    public function render()
    {
        $clients = Auth::user()->role === 'super_admin'
            ? Client::all() // Si es super_admin, cargar todos los clientes
            : Client::whereHas('user', function ($query) {
                $query->where('company_id', Auth::user()->company_id);
            })->get();

        return view('livewire.dashboard.payment.save', compact('clients'))->layout('layouts.app');
    }

    public function submit()
    {
        $this->validate();

        if ($this->payment) {
            $this->payment->update([
                'client_id' => $this->client_id,
                'amount_paid' => $this->amount_paid,
                'payment_date' => $this->payment_date,
                'paymet_detail' => $this->paymet_detail,
            ]);
            session()->flash('success', 'Pago actualizado exitosamente.');
        } else {
            Payment::create([
                'client_id' => $this->client_id,
                'amount_paid' => $this->amount_paid,
                'payment_date' => $this->payment_date,
                'paymet_detail' => $this->paymet_detail,
            ]);
            session()->flash('success', 'Pago registrado exitosamente.');
        }

        return redirect()->route('d-payment-index', [$this->client_id]);
    }
}
